/** @file   bonusobject.h
 * @brief   Declaration of BonusObject - class.
 * @version $Revision: 1.3 $
 * @date    $Date: 2006/04/12 21:10:24 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_BONUSOBJECT_H
#define H_WWW_BONUSOBJECT_H

#include "eng2d.h"

namespace WeWantWar {


/** @class  BonusObject
 * @brief   Represents the powerup bonuses being found from the playfield.
 * @author  Tomi Lamminsaari
 *
 * The pickable bonuses are instances of this class.
 */
class BonusObject
{
public:
  ///
  /// Static members and methods
  /// ==========================
  
  /** Types of the collectable bonus items. This constants also operate
   * as a indices to the vector holding the images of the bonus items.
   */
  typedef int Type;
  /** Rifle-bonus id */
  static const Type RIFLE = 0;
  /** Shotgun-bonus id */
  static const Type SHOTGUN = 1;
  /** Sniper rifle bonus. */
  static const Type SNIPERRIFLE = 2;
  /** Grenade bonus id */
  static const Type GRENADE = 3;
  /** RocketLauncher */
  static const Type ROCKETLAUNCHER = 4;
  /** First aid bonus id */
  static const Type FIRSTAID = 5;
  /** Flamethrower id */
  static const Type FLAMETHROWER = 6;
  /** Minigun-bonus id */
  static const Type MINIGUN = 7;
  /** Uzi-bonus id */
  static const Type UZI = 8;
  /** Extra life */
  static const Type EXTRALIFE = 9;
  /** 100 points */
  static const Type SCORE_100PTS = 10;
  /** 200 points */
  static const Type SCORE_200PTS = 11;
  /** Bodyarmor */
  static const Type BODYARMOR = 12;
  /** Unknown bonus type. */
  static const Type UNKNOWN = -1;
  
  
  /** Converts the given string to bonusitem. In level-files we need to
   * define the bonuses and it's more convinient to use text-format of the
   * bonusnames instead of numeric values.
   *
   * Supported strings are: rifle, shotgun, grenade, firstaid, flamethrower
   * minigun, uzi
   *
   * @param     bn                The bonus string
   * @return    Type of the bonus or BonusObject::UNKOWN if no match was found.
   */
  static Type stringToType( const std::string& bn );
  
  
  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Constructs new BonusObject.
   * @param     t                 Type of the bonus object.
   */
  BonusObject(Type t);
  
  /** Copy constructor.
   */
  BonusObject(const BonusObject& rO);
  
  /** Destructor.
   */
  virtual ~BonusObject();
  
  /** Operator =
   */
  BonusObject& operator = (const BonusObject& rO);


  ///
  /// Methods
  /// =======

  /** Sets the position where this bonus object will be located.
   * @param     p                 The location of this bonus (in pixels)
   */
  void position( const eng2d::Vec2D& p );
  
  /** Sets the size of the boundingsphere.
   * @param     s                 Radius of this bonus object.
   */
  void size( float s );
  
  /** Redraws this bonusobject.
   * @param     pTarget           Pointer to target bitmap.
   */
  virtual void redraw( BITMAP* pTarget );
  
  /** Updates this bonusobject
   */
  virtual void update();
  

  ///
  /// Getter methods
  /// ==================
  
  /** Returns the type of this bonus.
   * @return    Type
   */
  Type          type() const;
  
  /** Returns the position of this bonus.
   * @return    Position in pixels.
   */
  eng2d::Vec2D  position() const;
  
  /** Returns the radius of the bounding sphere.
   * @return    Radius of the bounding sphere.
   */
  float         size() const;

protected:

  static const int spriteIndexArray[13];

  ///
  /// Members
  /// =======

  /** THe type of this bonus. */
  Type          m_type;
  /** Position of this bonus. */
  eng2d::Vec2D  m_position;
  /** Radius of the bounding sphere. */
  float         m_size;
};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: bonusobject.h,v $
 * Revision 1.3  2006/04/12 21:10:24  lamminsa
 * Bodyarmor bonus added.
 *
 * Revision 1.2  2006/02/08 22:02:55  lamminsa
 * no message
 *
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.1  2005-11-13 14:31:13+02  lamminsa
 * RocketLauncher bonus was added.
 *
 */
 
